<?php
/*
 * Created by Artureanec
*/

namespace AgrariUm\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Scheme_Color;
use Elementor\Scheme_Typography;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class AgrariUm_Image_Widget extends Widget_Base {

    public function get_name() {
        return 'agrarium_image';
    }

    public function get_title() {
        return esc_html__('Image', 'agrarium_plugin');
    }

    public function get_icon() {
        return 'eicon-image';
    }

    public function get_categories() {
        return ['agrarium_widgets'];
    }

    protected function register_controls() {

        // ----------------------------- //
        // ---------- Content ---------- //
        // ----------------------------- //
        $this->start_controls_section(
            'section_image',
            [
                'label' => esc_html__('Image', 'agrarium_plugin')
            ]
        );

        $this->add_control(
            'image',
            [
                'label'     => esc_html__('Image', 'agrarium_plugin'),
                'type'      => Controls_Manager::MEDIA,
                'default'   => [
                    'url'       => Utils::get_placeholder_image_src(),
                ]
            ]
        );

        $this->add_control(
            'image_style',
            [
                'label'     => esc_html__('Image Style', 'agrarium_plugin'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'default',
                'options'   => [
                    'default'   => esc_html__('Default', 'agrarium_plugin'),
                    'framed'    => esc_html__('Framed', 'agrarium_plugin')
                ]
            ]
        );

        $this->add_control(
            'retina_ready',
            [
                'label'         => esc_html__('Retina Ready', 'agrarium_plugin'),
                'type'          => Controls_Manager::SWITCHER,
                'default'       => 'no',
                'return_value'  => 'yes',
                'label_off'     => esc_html__('No', 'agrarium_plugin'),
                'label_on'      => esc_html__('Yes', 'agrarium_plugin')
            ]
        );

        $this->add_control(
            'label',
            [
                'label'         => esc_html__('Label', 'agrarium_plugin'),
                'type'          => Controls_Manager::TEXT,
                'placeholder'   => esc_html__('Image Label', 'agrarium_plugin')
            ]
        );

        $this->add_responsive_control(
            'image_align',
            [
                'label'     => esc_html__('Image Alignment', 'agrarium_plugin'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'left'      => [
                        'title'     => esc_html__('Left', 'agrarium_plugin'),
                        'icon'      => 'fa fa-align-left',
                    ],
                    'center'    => [
                        'title'     => esc_html__('Center', 'agrarium_plugin'),
                        'icon'      => 'fa fa-align-center',
                    ],
                    'right'     => [
                        'title'     => esc_html__('Right', 'agrarium_plugin'),
                        'icon'      => 'fa fa-align-right',
                    ]
                ],
                'default'   => 'center',
                'selectors' => [
                    '{{WRAPPER}} .image-widget' => 'text-align: {{VALUE}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'image_float',
            [
                'label'     => esc_html__('Image Float', 'agrarium_plugin'),
                'type'      => Controls_Manager::CHOOSE,
                'options'   => [
                    'left'      => [
                        'title'     => esc_html__('Left', 'agrarium_plugin'),
                        'icon'      => 'fas fa-outdent',
                    ],
                    'none'      => [
                        'title'     => esc_html__('None', 'agrarium_plugin'),
                        'icon'      => 'fas fa-align-justify',
                    ],
                    'right'     => [
                        'title'     => esc_html__('Right', 'agrarium_plugin'),
                        'icon'      => 'fas fa-indent',
                    ]
                ],
                'default'   => 'none',
                'selectors' => [
                    '{{WRAPPER}} .image-container' => 'float: {{VALUE}};',
                ],
            ]
        );

        $this->end_controls_section();

        // ------------------------------------ //
        // ---------- Image Settings ---------- //
        // ------------------------------------ //
        $this->start_controls_section(
            'section_heading_settings',
            [
                'label' => esc_html__('Image Settings', 'agrarium_plugin'),
                'tab'   => Controls_Manager::TAB_STYLE
            ]
        );
        $this->add_control(
            'use_scroll_motion',
            [
                'label'         => esc_html__( 'Scroll Motion Effect', 'agrarium_plugin' ),
                'type'          => Controls_Manager::SWITCHER,
                'label_on'      => esc_html__( 'On', 'agrarium_plugin' ),
                'label_off'     => esc_html__( 'Off', 'agrarium_plugin' ),
                'return_value'  => 'yes',
                'default'       => 'no',
                'prefix_class'  => 'scroll-motion-'
            ]
        );
        $this->add_control(
            'scroll_direction',
            [
                'label'     => esc_html__('Scroll Direction', 'agrarium_plugin'),
                'type'      => Controls_Manager::SELECT,
                'default'   => 'up',
                'options'   => [
                    'up'   => esc_html__('Up', 'agrarium_plugin'),
                    'down'   => esc_html__('Down', 'agrarium_plugin')
                ],
                'condition' => [
                    'use_scroll_motion' => 'yes'
                ],
                'separator'     => 'after',
                'prefix_class'  => 'scroll-motion-direction-'
            ]
        );
        $this->add_responsive_control(
            'max_width',
            [
                'label'             => esc_html__( 'Max Width', 'agrarium_plugin' ),
                'type'              => Controls_Manager::SLIDER,
                'default'           => [
                    'unit'  => '%',
                ],
                'tablet_default'    => [
                    'unit'  => '%',
                ],
                'mobile_default'    => [
                    'unit'  => '%',
                ],
                'size_units'        => [ '%', 'px', 'vw' ],
                'range'             => [
                    '%'     => [
                        'min'   => 1,
                        'max'   => 200,
                    ],
                    'px'    => [
                        'min'   => 1,
                        'max'   => 1000,
                    ],
                    'vw'    => [
                        'min'   => 1,
                        'max'   => 100,
                    ],
                ],
                'selectors'         => [
                    '{{WRAPPER}} img.main-image' => 'max-width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->add_responsive_control(
            'frame_width',
            [
                'label'             => esc_html__( 'Frame Width', 'agrarium_plugin' ),
                'type'              => Controls_Manager::SLIDER,
                'default'           => [
                    'unit'  => '%',
                ],
                'tablet_default'    => [
                    'unit'  => '%',
                ],
                'mobile_default'    => [
                    'unit'  => '%',
                ],
                'size_units'        => [ '%', 'px', 'vw' ],
                'range'             => [
                    '%'     => [
                        'min'   => 1,
                        'max'   => 100,
                    ],
                    'px'    => [
                        'min'   => 1,
                        'max'   => 2000,
                    ],
                    'vw'    => [
                        'min'   => 1,
                        'max'   => 100,
                    ],
                ],
                'selectors'         => [
                    '{{WRAPPER}} .image-style-framed:before' => 'width: calc({{SIZE}}{{UNIT}} - 20px);',
                ],
                'condition' => [
                    'image_style'   => 'framed'
                ]
            ]
        );

        $this->add_control(
            'frame_color',
            [
                'label'     => esc_html__('Frame Color', 'agrarium_plugin'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .image-container.image-style-framed:before' => 'border-color: {{VALUE}};'
                ],
                'condition' => [
                    'image_style' => 'framed'
                ],
                'separator' => 'after'
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name'      => 'label_typography',
                'label'     => esc_html__('Label Typography', 'agrarium_plugin'),
                'selector'  => '{{WRAPPER}} .label'
            ]
        );

        $this->add_control(
            'label_color',
            [
                'label'     => esc_html__('Label Color', 'agrarium_plugin'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .label' => 'color: {{VALUE}};'
                ]
            ]
        );

        $this->add_control(
            'label_bg_color',
            [
                'label'     => esc_html__('Label Background', 'agrarium_plugin'),
                'type'      => Controls_Manager::COLOR,
                'selectors' => [
                    '{{WRAPPER}} .label' => 'background-color: {{VALUE}};'
                ]
            ]
        );

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings();

        $image          = $settings['image'];
        $retina_ready   = $settings['retina_ready'];
        $label          = $settings['label'];
        $image_style    = $settings['image_style'];

        // ------------------------------------ //
        // ---------- Widget Content ---------- //
        // ------------------------------------ //

        $image_src      = $image['url'];
        $image_meta     = agrarium_get_attachment_meta($image['id']);
        $image_alt_text = $image_meta['alt'];

        $img_wp_meta    = wp_get_attachment_metadata($image['id']);
        $width          = intval($img_wp_meta['width'] / 2);
        $height         = intval($img_wp_meta['height'] / 2);
        ?>
        <div class="image-widget">
            <div class="image-container<?php echo ( !empty($image_style) ? ' image-style-' . esc_attr($image_style) : ''); ?>">
                <img class="main-image" src="<?php echo esc_url($image_src); ?>" alt="<?php echo esc_attr($image_alt_text); ?>"<?php echo ( $retina_ready == 'yes' ? ' width="' . esc_attr($width) . '" height="' . esc_attr($height) . '"' : ''); ?> />
                <?php
                    if ( !empty($label) ) {
                        echo '<div class="label">' . esc_html($label) . '</div>';
                    }
                ?>
            </div>
        </div>
        <?php
    }

    protected function content_template() {}

    public function render_plain_content() {}
}
