<?php
/*
 * Created by Artureanec
*/

namespace AgrariUm\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Scheme_Color;
use Elementor\Scheme_Typography;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\REPEATER;
use Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

class AgrariUm_Content_Slider_Widget extends Widget_Base {

    public function get_name() {
        return 'agrarium_content_slider';
    }

    public function get_title() {
        return esc_html__('Content Slider', 'agrarium_plugin');
    }

    public function get_icon() {
        return 'eicon-post-slider';
    }

    public function get_categories() {
        return ['agrarium_widgets'];
    }

    public function get_script_depends() {
        return ['content_slider_widget'];
    }

    protected function register_controls() {

        // ----------------------------- //
        // ---------- Content ---------- //
        // ----------------------------- //
        $this->start_controls_section(
            'section_content',
            [
                'label' => esc_html__('Content Slider', 'agrarium_plugin')
            ]
        );

        $this->add_control(
            'content_width_type',
            [
                'label' => esc_html__('Content Width Type', 'agrarium_plugin'),
                'type' => Controls_Manager::SELECT,
                'default' => 'full',
                'options' => [
                    'boxed' => esc_html__('Boxed', 'agrarium_plugin'),
                    'full' => esc_html__('FullWidth', 'agrarium_plugin'),
                ],
                'separator' => 'before'
            ]
        );

        $this->add_responsive_control(
            'slider_height',
            [
                'label'     => esc_html__('Slider Height', 'agrarium_plugin'),
                'type'      => Controls_Manager::SLIDER,
                'default'   => [
                    'size'      => '973',
                    'unit'      => 'px'
                ],
                'range'     => [
                    'px'        => [
                        'min'       => 10,
                        'max'       => 2000,
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} .agrarium_content_slide' => 'height: {{SIZE}}{{UNIT}};'
                ],
                'separator' => 'before'
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'slide_name',
            [
                'label' => esc_html__('Slide Name', 'agrarium_plugin'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'separator' => 'after'
            ]
        );

        $repeater->add_control(
            'slide_type',
            [
                'label' => esc_html__('Slide Type', 'agrarium_plugin'),
                'type' => Controls_Manager::SELECT,
                'default' => 'fade_in_right',
                'options' => [
                    'fade_in_right' => esc_html__('Fade In Right', 'agrarium_plugin'),
                    'fade_in_left' => esc_html__('Fade In Left', 'agrarium_plugin'),
                    'corners_to_center' => esc_html__('Corners To Center', 'agrarium_plugin')
                ],
                'separator' => 'after'
            ]
        );



        $repeater->start_controls_tabs('button_settings_tabs');

            // -------------------- //
            // ------ BG Tab ------ //
            // -------------------- //
            $repeater->start_controls_tab(
                'tab_bg',
                [
                    'label' => esc_html__('BG', 'agrarium_plugin')
                ]
            );

                $repeater->add_group_control(
                    Group_Control_Background::get_type(),
                    [
                        'name' => 'background',
                        'label' => esc_html__( 'Background', 'agrarium_plugin' ),
                        'types' => [ 'classic' ],
                        'selector' => '{{WRAPPER}} {{CURRENT_ITEM}}'
                    ]
                );

            $repeater->end_controls_tab();

            // ----------------------- //
            // ------ Image Tab ------ //
            // ----------------------- //
            $repeater->start_controls_tab(
                'tab_image',
                [
                    'label' => esc_html__('Image', 'agrarium_plugin')
                ]
            );

                $repeater->add_control(
                    'active_image',
                    [
                        'label' => esc_html__('Active Image', 'agrarium_plugin'),
                        'type' => Controls_Manager::MEDIA,
                        'default' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'condition' => [
                            'slide_type' => ['fade_in_right', 'fade_in_left']
                        ]
                    ]
                );

                $repeater->add_responsive_control(
                    'image_margin',
                    [
                        'label' => esc_html__('Position', 'agrarium_plugin'),
                        'type' => Controls_Manager::DIMENSIONS,
                        'size_units' => ['%', 'px'],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .active_image' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ],
                        'condition' => [
                            'slide_type' => ['fade_in_right', 'fade_in_left']
                        ]
                    ]
                );

                $repeater->add_control(
                    'active_image_1',
                    [
                        'label' => esc_html__('Top Left Corner Image', 'agrarium_plugin'),
                        'type' => Controls_Manager::MEDIA,
                        'default' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'condition' => [
                            'slide_type' => 'corners_to_center'
                        ]
                    ]
                );

                $repeater->add_control(
                    'active_image_2',
                    [
                        'label' => esc_html__('Top Right Corner Image', 'agrarium_plugin'),
                        'type' => Controls_Manager::MEDIA,
                        'default' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'condition' => [
                            'slide_type' => 'corners_to_center'
                        ]
                    ]
                );

                $repeater->add_control(
                    'active_image_3',
                    [
                        'label' => esc_html__('Bottom Right Corner Image', 'agrarium_plugin'),
                        'type' => Controls_Manager::MEDIA,
                        'default' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'condition' => [
                            'slide_type' => 'corners_to_center'
                        ]
                    ]
                );

                $repeater->add_control(
                    'active_image_4',
                    [
                        'label' => esc_html__('Bottom Left Corner Image', 'agrarium_plugin'),
                        'type' => Controls_Manager::MEDIA,
                        'default' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'condition' => [
                            'slide_type' => 'corners_to_center'
                        ]
                    ]
                );

            $repeater->end_controls_tab();

            // ---------------------- //
            // ------ Icon Tab ------ //
            // ---------------------- //
            $repeater->start_controls_tab(
                'tab_icon',
                [
                    'label' => esc_html__('Icon', 'agrarium_plugin')
                ]
            );

                $repeater->add_control(
                    'icon_type',
                    [
                        'label' => esc_html__('Type of Icon', 'agrarium_plugin'),
                        'type' => Controls_Manager::SELECT,
                        'default' => 'none',
                        'options' => [
                            'none' => esc_html__('None', 'agrarium_plugin'),
                            'default' => esc_html__('Default Icon', 'agrarium_plugin'),
                            'svg' => esc_html__('SVG Icon', 'agrarium_plugin'),
                            'image' => esc_html__('Image Icon', 'agrarium_plugin')
                        ]
                    ]
                );

                $repeater->add_control(
                    'default_icon',
                    [
                        'label' => esc_html__('Icon', 'agrarium_plugin'),
                        'type' => Controls_Manager::ICONS,
                        'label_block' => true,
                        'default' => [
                            'value' => 'fas fa-star',
                            'library' => 'fa-solid'
                        ],
                        'condition' => [
                            'icon_type' => 'default'
                        ]
                    ]
                );

                $repeater->add_control(
                    'svg_icon',
                    [
                        'label' => esc_html__('SVG Icon', 'agrarium_plugin'),
                        'description' => esc_html__('Enter svg code', 'agrarium_plugin'),
                        'type' => Controls_Manager::TEXTAREA,
                        'default' => '',
                        'condition' => [
                            'icon_type' => 'svg'
                        ]
                    ]
                );

                $repeater->add_control(
                    'img_icon',
                    [
                        'label' => esc_html__('Image Icon', 'agrarium_plugin'),
                        'type' => Controls_Manager::MEDIA,
                        'default' => [
                            'url' => Utils::get_placeholder_image_src(),
                        ],
                        'condition' => [
                            'icon_type' => 'image'
                        ]
                    ]
                );

                $repeater->add_responsive_control(
                    'icon_size',
                    [
                        'label' => esc_html__('Icon Size', 'agrarium_plugin'),
                        'type' => Controls_Manager::SLIDER,
                        'range' => [
                            'px' => [
                                'min' => 5,
                                'max' => 200
                            ]
                        ],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .icon' => 'font-size: {{SIZE}}{{UNIT}};'
                        ],
                        'condition' => [
                            'icon_type' => 'default'
                        ]
                    ]
                );

                $repeater->add_responsive_control(
                    'icon_svg_size',
                    [
                        'label' => esc_html__('Icon Size', 'agrarium_plugin'),
                        'type' => Controls_Manager::SLIDER,
                        'range' => [
                            'px' => [
                                'min' => 5,
                                'max' => 200
                            ]
                        ],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .icon' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};'
                        ],
                        'condition' => [
                            'icon_type' => 'svg'
                        ]
                    ]
                );

                $repeater->add_responsive_control(
                    'icon_img_size',
                    [
                        'label' => esc_html__('Icon Size', 'agrarium_plugin'),
                        'type' => Controls_Manager::SLIDER,
                        'range' => [
                            'px' => [
                                'min' => 5,
                                'max' => 200
                            ]
                        ],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .icon' => 'width: {{SIZE}}{{UNIT}}; height: auto;'
                        ],
                        'condition' => [
                            'icon_type' => 'image'
                        ]
                    ]
                );

                $repeater->add_responsive_control(
                    'icon_margin',
                    [
                        'label' => esc_html__('Icon Margin', 'agrarium_plugin'),
                        'type' => Controls_Manager::DIMENSIONS,
                        'size_units' => ['px', '%'],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .icon' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ]
                    ]
                );

            $repeater->end_controls_tab();

            // ----------------------- //
            // ------ Title Tab ------ //
            // ----------------------- //
            $repeater->start_controls_tab(
                'tab_title',
                [
                    'label' => esc_html__('Title', 'agrarium_plugin')
                ]
            );

                $repeater->add_control(
                    'heading',
                    [
                        'label' => esc_html__('Title', 'agrarium_plugin'),
                        'type' => Controls_Manager::WYSIWYG,
                        'label_block' => true,
                        'placeholder' => esc_html__('Enter Title', 'agrarium_plugin'),
                        'default' => esc_html__('Title', 'agrarium_plugin')
                    ]
                );

                $repeater->add_control(
                    'heading_tag',
                    [
                        'label'     => esc_html__('HTML Tag', 'agrarium_plugin'),
                        'type'      => Controls_Manager::SELECT,
                        'options'   => [
                            'h1'        => esc_html__( 'H1', 'agrarium_plugin' ),
                            'h2'        => esc_html__( 'H2', 'agrarium_plugin' ),
                            'h3'        => esc_html__( 'H3', 'agrarium_plugin' ),
                            'h4'        => esc_html__( 'H4', 'agrarium_plugin' ),
                            'h5'        => esc_html__( 'H5', 'agrarium_plugin' ),
                            'h6'        => esc_html__( 'H6', 'agrarium_plugin' ),
                            'div'       => esc_html__( 'div', 'agrarium_plugin' ),
                            'span'      => esc_html__( 'span', 'agrarium_plugin' ),
                            'p'         => esc_html__( 'p', 'agrarium_plugin' )
                        ],
                        'default'   => 'div'
                    ]
                );
                $repeater->add_control(
                    'subheading_status',
                    [
                        'label' => esc_html__('Show Subheading', 'agrarium_plugin'),
                        'type' => Controls_Manager::SWITCHER,
                        'label_on' => esc_html__( 'Show', 'agrarium_plugin' ),
												'label_off' => esc_html__( 'Hide', 'agrarium_plugin' ),
												'return_value' => 'yes',
												'default' => 'no'
                    ]
                );
                $repeater->add_control(
                    'subheading',
                    [
                        'label' => esc_html__('Subheading', 'agrarium_plugin'),
                        'type' => Controls_Manager::TEXT,
                        'label_block' => true,
                        'placeholder' => esc_html__('Enter Subheading', 'agrarium_plugin'),
                        'default' => esc_html__('Subheading', 'agrarium_plugin'),
                        'condition' => [
                        	'subheading_status' => 'yes'
                        ]
                    ]
                );

                $repeater->add_group_control(
                    Group_Control_Typography::get_type(),
                    [
                        'name' => 'heading_typography',
                        'label' => esc_html__('Heading Typography', 'agrarium_plugin'),
                        'selector' => '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slider_title'
                    ]
                );

                $repeater->add_control(
                    'heading_color',
                    [
                        'label' => esc_html__('Heading Color', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slider_title' => 'color: {{VALUE}};'
                        ]
                    ]
                );
                $repeater->add_group_control(
                    Group_Control_Typography::get_type(),
                    [
                        'name' => 'subheading_typography',
                        'label' => esc_html__('Subheading Typography', 'agrarium_plugin'),
                        'selector' => '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slider_subheading',
                        'condition' => [
                        	'subheading_status' => 'yes'
                        ]
                    ]
                );

                $repeater->add_control(
                    'subheading_color',
                    [
                        'label' => esc_html__('Subheading Color', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slider_subheading' => 'color: {{VALUE}};'
                        ],
                        'condition' => [
                        	'subheading_status' => 'yes'
                        ]                        
                    ]
                );
                $repeater->add_responsive_control(
                    'space_between_subheading_title',
                    [
                        'label' => esc_html__('Space between subheading and title', 'agrarium_plugin'),
                        'type' => Controls_Manager::SLIDER,
                        'range' => [
                            'px' => [
                                'min' => 0,
                                'max' => 200
                            ]
                        ],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slider_subheading' => 'margin-bottom: {{SIZE}}{{UNIT}};'
                        ],
                        'condition' => [
                            'subheading_status' => 'yes'
                        ]
                    ]
                );

            $repeater->end_controls_tab();

            // ---------------------- //
            // ------ Text Tab ------ //
            // ---------------------- //
            $repeater->start_controls_tab(
                'tab_text',
                [
                    'label' => esc_html__('Text', 'agrarium_plugin')
                ]
            );

                $repeater->add_control(
                    'text',
                    [
                        'label' => esc_html__('Promo Text', 'agrarium_plugin'),
                        'type' => Controls_Manager::WYSIWYG,
                        'default' => '',
                        'placeholder' => esc_html__('Enter Promo Text', 'agrarium_plugin'),
                        'separator' => 'before'
                    ]
                );

                $repeater->add_group_control(
                    Group_Control_Typography::get_type(),
                    [
                        'name' => 'text_typography',
                        'label' => esc_html__('Text Typography', 'agrarium_plugin'),
                        'selector' => '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slider_promo_text'
                    ]
                );

                $repeater->add_control(
                    'text_color',
                    [
                        'label' => esc_html__('Text Color', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slider_promo_text' => 'color: {{VALUE}};'
                        ]
                    ]
                );

                $repeater->add_responsive_control(
                    'content_margin',
                    [
                        'label' => esc_html__('Content Margin', 'agrarium_plugin'),
                        'type' => Controls_Manager::DIMENSIONS,
                        'size_units' => ['px', '%'],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_container' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ],
                        'separator' => 'before'
                    ]
                );
                $repeater->add_responsive_control(
                    'content_padding',
                    [
                        'label' => esc_html__('Text Content Padding', 'agrarium_plugin'),
                        'type' => Controls_Manager::DIMENSIONS,
                        'size_units' => ['px', '%'],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_container .agrarium_slider_column:first-child' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ],
                        'separator' => 'before'
                    ]
                );

                $repeater->add_responsive_control(
                    'text_padding',
                    [
                        'label' => esc_html__('Promo Text Padding', 'agrarium_plugin'),
                        'type' => Controls_Manager::DIMENSIONS,
                        'size_units' => ['px', '%'],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slider_promo_text' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                        ]
                    ]
                );

            $repeater->end_controls_tab();

            // ------------------------ //
            // ------ Button Tab ------ //
            // ------------------------ //
            $repeater->start_controls_tab(
                'tab_button',
                [
                    'label' => esc_html__('Button', 'agrarium_plugin')
                ]
            );

                $repeater->add_control(
                    'button_text',
                    [
                        'label' => esc_html__('Button Text', 'agrarium_plugin'),
                        'type' => Controls_Manager::TEXT,
                        'default' => esc_html__('Button', 'agrarium_plugin'),
                        'separator' => 'before'
                    ]
                );

                $repeater->add_control(
                    'button_link',
                    [
                        'label' => esc_html__('Button Link', 'agrarium_plugin'),
                        'type' => Controls_Manager::URL,
                        'label_block' => true,
                        'default' => [
                            'url' => '',
                            'is_external' => 'true',
                        ],
                        'placeholder' => esc_html__( 'http://your-link.com', 'agrarium_plugin' )
                    ]
                );

                $repeater->add_control(
                    'button_color',
                    [
                        'label' => esc_html__('Button Color', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium-button' => 'color: {{VALUE}};'
                        ],
                        'separator' => 'before'
                    ]
                );

                $repeater->add_control(
                    'button_bd',
                    [
                        'label' => esc_html__('Button Border', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium-button' => 'border-color: {{VALUE}};'
                        ]
                    ]
                );

                $repeater->add_control(
                    'button_bg',
                    [
                        'label' => esc_html__('Button Background', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium-button' => 'background-color: {{VALUE}};'
                        ]
                    ]
                );
                $repeater->add_group_control(
                    Group_Control_Box_Shadow::get_type(),
                    [   
                        'name'      => 'button_box_shadow',
                        'label'     => esc_html__('Button Box Shadow', 'agrarium_plugin'),
                        'type'      => Controls_Manager::COLOR,
                        'selector'  => '{{WRAPPER}} .agrarium-button'
                    ]
                );
                $repeater->add_group_control(
                    Group_Control_Box_Shadow::get_type(),
                    [   
                        'name'      => 'button_box_shadow_hover',
                        'label'     => esc_html__('Button Hover Box Shadow', 'agrarium_plugin'),
                        'type'      => Controls_Manager::COLOR,
                        'selector'  => '{{WRAPPER}} .agrarium-button:hover'
                    ]
                );

                $repeater->add_control(
                    'button_color_hover',
                    [
                        'label' => esc_html__('Button Hover Color', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium-button:hover' => 'color: {{VALUE}};'
                        ],
                        'separator' => 'before'
                    ]
                );

                $repeater->add_control(
                    'button_bd_hover',
                    [
                        'label' => esc_html__('Button Border Hover', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium-button:hover' => 'border-color: {{VALUE}};'
                        ]
                    ]
                );

                $repeater->add_control(
                    'button_bg_hover',
                    [
                        'label' => esc_html__('Button Background Hover', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium-button:after' => 'background-color: {{VALUE}};'
                        ]
                    ]
                );

                $repeater->add_responsive_control(
                    'button_top_margin',
                    [
                        'label' => esc_html__('Button Top Space', 'agrarium_plugin'),
                        'type' => Controls_Manager::SLIDER,
                        'range' => [
                            'px' => [
                                'min' => 0,
                                'max' => 200
                            ]
                        ],
                        'selectors' => [
                            '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium-button' => 'margin-top: {{SIZE}}{{UNIT}};'
                        ]
                    ]
                );

            $repeater->end_controls_tab();

        $repeater->end_controls_tabs();

        $repeater->add_responsive_control(
            'content_max_width',
            [
                'label' => esc_html__('Content Container Max Width', 'agrarium_plugin'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%', 'px'],
                'range' => [
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 50,
                        'max' => 2000,
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_container' => 'max-width: {{SIZE}}{{UNIT}};'
                ],
                'separator' => 'before'
            ]
        );

        $repeater->add_responsive_control(
            'content_align',
            [
                'label' => esc_html__('Content Container Alignment', 'agrarium_plugin'),
                'type' => Controls_Manager::CHOOSE,
                'options' => [
                    'flex-start' => [
                        'title' => esc_html__( 'Left', 'agrarium_plugin' ),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'agrarium_plugin' ),
                        'icon' => 'eicon-h-align-center',
                    ],
                    'flex-end' => [
                        'title' => esc_html__( 'Right', 'agrarium_plugin' ),
                        'icon' => 'eicon-h-align-right',
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slide_wrapper' => 'justify-content: {{VALUE}};'
                ],
                'separator' => 'before'
            ]
        );

        $repeater->add_responsive_control(
            'content_v_align',
            [
                'label' => esc_html__('Content Container Vertical Alignment', 'agrarium_plugin'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'center',
                'options' => [
                    'flex-start' => [
                        'title' => esc_html__( 'Top', 'agrarium_plugin' ),
                        'icon' => 'eicon-v-align-top',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'agrarium_plugin' ),
                        'icon' => 'eicon-v-align-middle',
                    ],
                    'flex-end' => [
                        'title' => esc_html__( 'Bottom', 'agrarium_plugin' ),
                        'icon' => 'eicon-v-align-bottom',
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_slide_wrapper' => 'align-items: {{VALUE}};'
                ],
                'separator' => 'before'
            ]
        );

        $repeater->add_responsive_control(
            'content_text_align',
            [
                'label' => esc_html__('Content Container Text Align', 'agrarium_plugin'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'center',
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'agrarium_plugin' ),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'agrarium_plugin' ),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'agrarium_plugin' ),
                        'icon' => 'eicon-text-align-right',
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_container' => 'text-align: {{VALUE}};'
                ],
                'separator' => 'before'
            ]
        );
        $repeater->add_control(
            'slide_video_link',
            [
                'label' => esc_html__('Video Link', 'agrarium_plugin'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'placeholder' => esc_html__( 'http://your-link.com', 'agrarium_plugin' )
            ]
        );
        $repeater->add_responsive_control(
            'video_button_container_width',
            [
                'label' => esc_html__('Video Button Container Width', 'agrarium_plugin'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['%', 'px'],
                'range' => [
                    '%' => [
                        'min' => 1,
                        'max' => 100,
                    ],
                    'px' => [
                        'min' => 75,
                        'max' => 2000,
                    ]
                ],
                'default' => [
									'unit' => '%',
									'size' => 16.6666,
								],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_container.with_button .agrarium_slider_column:nth-child(2)' => 'width: {{SIZE}}{{UNIT}};'
                ],
                'separator' => 'before',
                'condition' => [	
                	'slide_video_link!' => ''
                ]
            ]
        );
        $repeater->add_control(
            'video_button_position',
            [
                'label' => esc_html__('Video Button Position', 'agrarium_plugin'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'right',
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'agrarium_plugin' ),
                        'icon' => 'eicon-h-align-left',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'agrarium_plugin' ),
                        'icon' => 'eicon-h-align-right',
                    ]
                ],
                'separator' => 'before',
                'condition' => [	
                	'slide_video_link!' => ''
                ]
            ]
        );
        $repeater->add_responsive_control(
            'video_button_text_align',
            [
                'label' => esc_html__('Video Button Alignment', 'agrarium_plugin'),
                'type' => Controls_Manager::CHOOSE,
                'default' => 'center',
                'options' => [
                    'left' => [
                        'title' => esc_html__( 'Left', 'agrarium_plugin' ),
                        'icon' => 'eicon-text-align-left',
                    ],
                    'center' => [
                        'title' => esc_html__( 'Center', 'agrarium_plugin' ),
                        'icon' => 'eicon-text-align-center',
                    ],
                    'right' => [
                        'title' => esc_html__( 'Right', 'agrarium_plugin' ),
                        'icon' => 'eicon-text-align-right',
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_container .slider_video_button' => 'text-align: {{VALUE}};'
                ],  
                'separator' => 'before',
                'condition' => [	
                	'slide_video_link!' => ''
                ]
            ]
        );
        $repeater->add_responsive_control(
	          'video_button_margin',
	          [
	              'label' => esc_html__('Video Button Margin', 'agrarium_plugin'),
	              'type' => Controls_Manager::DIMENSIONS,
	              'size_units' => ['px', '%'],
	              'selectors' => [
	                  '{{WRAPPER}} {{CURRENT_ITEM}} .agrarium_content_container .slider_video_button .agrarium_video_trigger_button' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
	              ],
	              'separator' => 'before',
                'condition' => [	
                	'slide_video_link!' => ''
                ]
	          ]
	      );

        $repeater->add_control(
            'divider_1',
            [
                'type' => Controls_Manager::DIVIDER,
            ]
        );

        $this->add_control(
            'slides',
            [
                'label' => esc_html__('Slides', 'agrarium_plugin'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'title_field' => '{{{slide_name}}}',
                'prevent_empty' => false,
                'separator' => 'before'
            ]
        );

        $this->end_controls_section();

        // ---------------------------- //
        // ---------- Slider ---------- //
        // ---------------------------- //
        $this->start_controls_section(
            'section_slider',
            [
                'label' => esc_html__('Slider Settings', 'agrarium_plugin')
            ]
        );

        $this->add_control(
            'speed',
            [
                'label' => esc_html__('Animation Speed', 'agrarium_plugin'),
                'type' => Controls_Manager::NUMBER,
                'default' => 1200,
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'infinite',
            [
                'label' => esc_html__('Infinite Loop', 'agrarium_plugin'),
                'type' => Controls_Manager::SELECT,
                'default' => 'yes',
                'options' => [
                    'yes' => esc_html__('Yes', 'agrarium_plugin'),
                    'no' => esc_html__('No', 'agrarium_plugin'),
                ],
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'autoplay',
            [
                'label' => esc_html__('Autoplay', 'agrarium_plugin'),
                'type' => Controls_Manager::SELECT,
                'default' => 'yes',
                'options' => [
                    'yes' => esc_html__('Yes', 'agrarium_plugin'),
                    'no' => esc_html__('No', 'agrarium_plugin'),
                ],
                'separator' => 'before'
            ]
        );

        $this->add_control(
            'autoplay_speed',
            [
                'label' => esc_html__('Autoplay Speed', 'agrarium_plugin'),
                'type' => Controls_Manager::NUMBER,
                'default' => 5000,
                'condition' => [
                    'autoplay' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'pause_on_hover',
            [
                'label' => esc_html__('Pause on Hover', 'agrarium_plugin'),
                'type' => Controls_Manager::SELECT,
                'default' => 'yes',
                'options' => [
                    'yes' => esc_html__('Yes', 'agrarium_plugin'),
                    'no' => esc_html__('No', 'agrarium_plugin'),
                ],
                'condition' => [
                    'autoplay' => 'yes'
                ]
            ]
        );

        $this->add_control(
            'rtl_support',
            [
                'label' => esc_html__('Rtl Support', 'agrarium_plugin'),
                'type' => Controls_Manager::SWITCHER,
                'default' => 'no',
                'label_off' => esc_html__('Off', 'agrarium_plugin'),
                'label_on' => esc_html__('On', 'agrarium_plugin'),
                'separator' => 'before'
            ]
        );
        $this->add_control(
            'show_arrows_status',
            [
                'label' => esc_html__('Show Arrows', 'agrarium_plugin'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'agrarium_plugin' ),
								'label_off' => esc_html__( 'Hide', 'agrarium_plugin' ),
								'return_value' => 'yes',
								'default' => 'no'
            ]
        );   
        $this->add_control(
            'show_dots_status',
            [
                'label' => esc_html__('Show Navigation Dots', 'agrarium_plugin'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'agrarium_plugin' ),
								'label_off' => esc_html__( 'Hide', 'agrarium_plugin' ),
								'return_value' => 'yes',
								'default' => 'yes'
            ]
        );

        $this->end_controls_section();

        // ------------------------------------------- //
        // ---------- Video Button Settings ---------- //
        // ------------------------------------------- //

        $this->start_controls_section(
            'section_video_settings',
            [
                'label' => esc_html__('Video Settings', 'agrarium_plugin')
            ]
        );
        $this->add_control(
            'show_video_button',
            [
                'label' => esc_html__('Show Video Button', 'agrarium_plugin'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'agrarium_plugin' ),
                'label_off' => esc_html__( 'Hide', 'agrarium_plugin' ),
                'return_value' => 'yes',
                'default' => 'no'
            ]
        );

        $this->add_control(
            'video_link',
            [
                'label' => esc_html__('Video Link', 'agrarium_plugin'),
                'type' => Controls_Manager::TEXT,
                'label_block' => true,
                'default' => 'https://www.youtube.com/embed/XHOmBV4js_E',
                'placeholder' => esc_html__( 'http://your-link.com', 'agrarium_plugin' ),
                'condition' => [
                    'show_video_button' => 'yes'
                ]
            ]
        );
        $this->add_control(
            'video_button_text',
            [
                'label' => esc_html__('Video Button Text', 'agrarium_plugin'),
                'type' => Controls_Manager::TEXT,
                'default' => esc_html__('Watch Our Video', 'agrarium_plugin'),
                'separator' => 'before',
                'condition' => [
                    'show_video_button' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        // ------------------------------------------- //
        // ---------- Additional Settings ---------- //
        // ------------------------------------------- //
        $this->start_controls_section(
            'additional_settings',
            [
                'label' => esc_html__('Additional Settings', 'agrarium_plugin')
            ]
        );

        $this->add_control(
            'show_phone_status',
            [
                'label' => esc_html__('Show Phone Number', 'agrarium_plugin'),
                'description' => 'Will be shown phone number from top bar settings',
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'agrarium_plugin' ),
                'label_off' => esc_html__( 'Hide', 'agrarium_plugin' ),
                'return_value' => 'yes',
                'default' => 'no'
            ]
        );

        $this->add_control(
            'contact_phone_title',
            [
                'label'         => esc_html__('Phone title', 'agrarium_plugin'),
                'type'          => Controls_Manager::TEXT,
                'label_block'   => true,
                'default'       => 'Have any Questions?',
                'condition'     => [
                    'show_phone_status'  => 'yes'
                ]
            ]
        );

        $this->add_control(
            'show_email_status',
            [
                'label' => esc_html__('Show Email', 'agrarium_plugin'),
                'description' => 'Will be shown email from top bar settings',
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'agrarium_plugin' ),
                'label_off' => esc_html__( 'Hide', 'agrarium_plugin' ),
                'return_value' => 'yes',
                'default' => 'no'
            ]
        );

        $this->add_control(
            'contact_email_title',
            [
                'label'         => esc_html__('Email title', 'agrarium_plugin'),
                'type'          => Controls_Manager::TEXT,
                'label_block'   => true,
                'default'       => 'Email us',
                'condition'     => [
                    'show_phone_status'  => 'yes'
                ]
            ]
        );

        $this->add_responsive_control(
            'bottom_bar_position',
            [
                'label' => esc_html__('Bottom Bar Position', 'agrarium_plugin'),
                'type' => Controls_Manager::SLIDER,
                'default'   => [
                    'size'      => '-10',
                    'unit'      => 'px'
                ],
                'range'     => [
                    'px'        => [
                        'min'       => -100,
                        'max'       => 100,
                    ]
                ],
                'selectors' => [
                    '{{WRAPPER}} .agrarium_content_slider_bar' => 'bottom: {{SIZE}}{{UNIT}};'
                ],
                'separator' => 'before'
            ]
        );
        $this->add_control(
            'show_scrolldown_status',
            [
                'label' => esc_html__('Show ScrollDown Button', 'agrarium_plugin'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'agrarium_plugin' ),
                'label_off' => esc_html__( 'Hide', 'agrarium_plugin' ),
                'return_value' => 'yes',
                'default' => 'yes'
            ]
        );
        $this->add_control(
            'scrolldown_link',
            [
                'label' => esc_html__('Scroll Section ID', 'agrarium_plugin'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => 'section_id',
                'condition' => [
                    'show_scrolldown_status' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        // ------------------------------------------- //
        // ---------- Button Style Settings ---------- //
        // ------------------------------------------- //
        $this->start_controls_section(
            'section_button_settings',
            [
                'label' => esc_html__('Content Settings', 'agrarium_plugin'),
                'tab' => Controls_Manager::TAB_STYLE
            ]
        );

        $this->add_group_control(
            Group_Control_Typography::get_type(),
            [
                'name' => 'button_typography',
                'label' => esc_html__('Button Typography', 'agrarium_plugin'),
                'selector' => '{{WRAPPER}} .agrarium-button'
            ]
        );

        $this->add_control(
            'button_radius',
            [
                'label' => esc_html__('Border Radius', 'agrarium_plugin'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .agrarium-button' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]
        );

        $this->add_responsive_control(
            'button_padding',
            [
                'label' => esc_html__('Button Padding', 'agrarium_plugin'),
                'type' => Controls_Manager::DIMENSIONS,
                'size_units' => ['px', '%'],
                'selectors' => [
                    '{{WRAPPER}} .agrarium-button' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};'
                ]
            ]
        );
        $this->add_control(
            'show_slider_overlay',
            [
                'label' => esc_html__('Show Slider Bottom Overlay', 'agrarium_plugin'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => esc_html__( 'Show', 'agrarium_plugin' ),
                'label_off' => esc_html__( 'Hide', 'agrarium_plugin' ),
                'return_value' => 'yes',
                'default' => 'yes'
            ]
        );

        $this->add_control(
            'slider_overlay_image',
            [
                'label' => esc_html__('Image Overlay', 'agrarium_plugin'),
                'type' => Controls_Manager::MEDIA,
                'default' => [
                    'url' => Utils::get_placeholder_image_src(),
                ],
                'condition' => [
                    'show_slider_overlay' => 'yes'
                ]
            ]
        );

        $this->end_controls_section();

        // ------------------------------------------- //
        // ---------- Button Style Settings ---------- //
        // ------------------------------------------- //
        $this->start_controls_section(
            'section_slider_settings',
            [
                'label' => esc_html__('Slider Settings', 'agrarium_plugin'),
                'tab' => Controls_Manager::TAB_STYLE,
                'condition' => [
		              'show_arrows_status' => 'yes'
		          ]
            ]
        );            

        $this->start_controls_tabs('button_settings_tabs');

            // ------------------------ //
            // ------ Normal Tab ------ //
            // ------------------------ //
            $this->start_controls_tab(
                'tab_arrows_normal',
                [
                    'label' => esc_html__('Normal', 'agrarium_plugin')
                ]
            );

                $this->add_control(
                    'nav_color',
                    [
                        'label' => esc_html__('Slider Arrows Color', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .slick-arrow' => 'color: {{VALUE}};'
                        ]
                    ]
                );

                $this->add_control(
                    'nav_bg',
                    [
                        'label' => esc_html__('Slider Arrows Background', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .slick-arrow' => 'background: {{VALUE}};'
                        ]
                    ]
                );

            $this->end_controls_tab();

            // ----------------------- //
            // ------ Hover Tab ------ //
            // ----------------------- //
            $this->start_controls_tab(
                'tab_arrows_hover',
                [
                    'label' => esc_html__('Hover', 'agrarium_plugin')
                ]
            );

                $this->add_control(
                    'nav_hover',
                    [
                        'label' => esc_html__('Slider Arrows Hover', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .slick-arrow:not(.slick-disabled):hover' => 'color: {{VALUE}};'
                        ]
                    ]
                );

                $this->add_control(
                    'nav_bg_hover',
                    [
                        'label' => esc_html__('Slider Arrows Hover Background', 'agrarium_plugin'),
                        'type' => Controls_Manager::COLOR,
                        'selectors' => [
                            '{{WRAPPER}} .slick-arrow:not(.slick-disabled):hover' => 'background: {{VALUE}};'
                        ]
                    ]
                );

            $this->end_controls_tab();

        $this->end_controls_tabs();

        $this->end_controls_section();
    }

    protected function render() {
        $settings = $this->get_settings();
        $content_width_type = $settings['content_width_type'];

        $contact_phone_title = $settings['contact_phone_title'];
        $contact_email_title = $settings['contact_email_title'];

        $slides = $settings['slides'];

        $slider_options = [
        	'arrows'        => ('yes' === $settings['show_arrows_status']),
        	'dots'          => ('yes' === $settings['show_dots_status']),
            'pauseOnHover'  => ('yes' === $settings['pause_on_hover']),
            'autoplay'      => ('yes' === $settings['autoplay']),
            'infinite'      => ('yes' === $settings['infinite']),
            'speed'         => absint($settings['speed']),
            'rtl'           => ( $settings['rtl_support'] == 'yes' ? true : false )
        ];        

        // ------------------------------------ //
        // ---------- Widget Content ---------- //
        // ------------------------------------ //
        ?>

        <div class="agrarium_content_slider_widget" <?php echo $settings['rtl_support'] == 'yes' ? 'dir="rtl"' : '' ?>>
            <div class="agrarium_content_slider_wrapper">
                <div class="agrarium_content_slider agrarium_slider_slick" data-slider-options="<?php echo esc_attr(wp_json_encode($slider_options)); ?>">
                    <?php
                    foreach ($slides as $slide) {
                        echo '<div class="agrarium_content_slide elementor-repeater-item-' . esc_attr($slide['_id']) . ' slide-animation-type-' . esc_attr($slide['slide_type']) . '">';
                            if ( $slide['slide_type'] == 'corners_to_center' ) {
                                $image_width_px_1   = wp_get_attachment_image_src($slide['active_image_1']['id'], 'full')[1];
                                $image_width_px_2   = wp_get_attachment_image_src($slide['active_image_2']['id'], 'full')[1];
                                $image_width_px_3   = wp_get_attachment_image_src($slide['active_image_3']['id'], 'full')[1];
                                $image_width_px_4   = wp_get_attachment_image_src($slide['active_image_4']['id'], 'full')[1];
                                $width_1            = ( (int)$image_width_px_1 / 1920 ) * 100;
                                $width_2            = ( (int)$image_width_px_2 / 1920 ) * 100;
                                $width_3            = ( (int)$image_width_px_3 / 1920 ) * 100;
                                $width_4            = ( (int)$image_width_px_4 / 1920 ) * 100;
                                $image_width_perc_1 = round($width_1, 4);
                                $image_width_perc_2 = round($width_2, 4);
                                $image_width_perc_3 = round($width_3, 4);
                                $image_width_perc_4 = round($width_4, 4);
                                echo ( !empty($slide['active_image_1']) ? '<div class="active_image active_image_left_top" style="width: ' . esc_attr($image_width_perc_1) . '%;">' . wp_get_attachment_image( $slide['active_image_1']['id'], 'full' ) . '</div>' : '' );
                                echo ( !empty($slide['active_image_2']) ? '<div class="active_image active_image_right_top" style="width: ' . esc_attr($image_width_perc_2) . '%;">' . wp_get_attachment_image( $slide['active_image_2']['id'], 'full' ) . '</div>' : '' );
                                echo ( !empty($slide['active_image_3']) ? '<div class="active_image active_image_right_bottom" style="width: ' . esc_attr($image_width_perc_3) . '%;">' . wp_get_attachment_image( $slide['active_image_3']['id'], 'full' ) . '</div>' : '' );
                                echo ( !empty($slide['active_image_4']) ? '<div class="active_image active_image_left_bottom" style="width: ' . esc_attr($image_width_perc_4) . '%;">' . wp_get_attachment_image( $slide['active_image_4']['id'], 'full' ) . '</div>' : '' );
                            } else {
	                        		if(!empty($slide['active_image']['id'])) {
	                        			$image_width_px     = wp_get_attachment_image_src($slide['active_image']['id'], 'full')[1];
		                                $width = ( (int)$image_width_px / 1920 ) * 100;
		                                $image_width_perc   = round($width, 4);
	                                	echo '<div class="active_image" style="width: ' . esc_attr($image_width_perc) . '%;">' . wp_get_attachment_image( $slide['active_image']['id'], 'full' ) . '</div>';
	                        		}                                
                            }
                                $show_button = !empty($slide['slide_video_link']);
                                $container_class = 'container' . ($content_width_type === 'boxed' ? '-boxed' : '');
                                $content_container_class = ($show_button ? ' with_button' : '');
                                if($show_button) {
                                	$content_container_class .= " video_button_position_{$slide['video_button_position']}";
                                }                                
                                echo '<div class="' . $container_class . ' agrarium_full_cont">';
                                  echo '<div class="agrarium_full_cont">';
                                      echo '<div class="agrarium_content_slide_wrapper">';
                                          echo '<div class="agrarium_content_container' . $content_container_class . '">';
                                          	echo '<div class="agrarium_slider_column">';

                                              if ($slide['icon_type'] == 'default') {
                                                  echo '<div class="agrarium_content_wrapper_1">';
                                                      echo '<i class="icon ' . esc_attr($slide['default_icon']['value']) . '"></i>';
                                                  echo '</div>';
                                              }
                                              if ($slide['icon_type'] == 'svg') {
                                                  echo '<div class="agrarium_content_wrapper_1">';
                                                      echo '<span class="icon">' . organium_output_code($slide['svg_icon']) . '</span>';
                                                  echo '</div>';
                                              }
                                              if ($slide['icon_type'] == 'image') {
                                                  echo '<div class="agrarium_content_wrapper_1">';
                                                      echo '<span class="icon">' . wp_get_attachment_image( $slide['img_icon']['id'], 'full' ) . '</span>';
                                                  echo '</div>';
                                              }
                                              if ( !empty($slide['heading']) ) {
                                                  echo '<div class="agrarium_content_wrapper_2">';
                                                  		if($slide['subheading_status'] == 'yes' && !empty($slide['subheading'])) {
                                                  			echo '<div class="agrarium_content_slider_subheading">';
                                                         	echo agrarium_output_code($slide['subheading']);
                                                      	echo '</div>';
                                                  		}
                                                      echo '<' . esc_html($slide['heading_tag']) . ' class="agrarium_content_slider_title">';
                                                        echo wp_kses($slide['heading'], array(
                                                            'br'        => array(),
                                                            'span'      => array(
                                                                'style'     => true
                                                            ),
                                                            'a'         => array(
                                                                'href'      => true,
                                                                'target'    => true
                                                            ),
                                                            'img'       => array(
                                                                'src'       => true,
                                                                'srcset'    => true,
                                                                'sizes'     => true,
                                                                'class'     => true,
                                                                'alt'       => true,
                                                                'title'     => true
                                                            ),
                                                            'em'        => array(),
                                                            'strong'    => array(),
                                                            'del'       => array()
                                                        ));
                                                      echo '</' . esc_html($slide['heading_tag']) . '>';
                                                  echo '</div>';
                                              }

                                              if ($slide['text'] !== '') {
                                                  echo '<div class="agrarium_content_wrapper_3">';
                                                      echo '<div class="agrarium_content_slider_promo_text">' . agrarium_output_code($slide['text']) . '</div>';
                                                  echo '</div>';
                                              }

                                              if ($slide['button_text'] !== '') {
                                                  if ($slide['button_link']['url'] !== '') {
                                                      $button_url = $slide['button_link']['url'];
                                                  } else {
                                                      $button_url = '#';
                                                  }
                                                  echo '<div class="agrarium_content_wrapper_4">';
                                                      echo '<a class="agrarium-button" href="' . esc_url($button_url) . '"' . (($slide['button_link']['is_external'] == true) ? ' target="_blank"' : '') . (($slide['button_link']['nofollow'] == 'on') ? ' rel="nofollow"' : '') . '>';
                                                          echo esc_html($slide['button_text']);
                                                      echo '</a>';
                                                  echo '</div>';
                                              }
                                             echo '</div>';
                                             if($slide['slide_video_link'] !== '') { ?>
                                             		<div class="agrarium_slider_column">
                                             			<div class="agrarium_video_widget slider_video_button">
												            <div class="agrarium_preview_container">   
												                <a class="agrarium_video_trigger_button" href="<?php echo esc_js('javascript:void(0)'); ?>">
											                		<span class="agrarium_video_trigger_button_inner">
											                			<span class="agrarium_button_icon"><i class="icomoon-play"></i></span>
											                		</span>             
												                </a>
												            </div>
    										                <div class="agrarium_video_container">
    										                    <div class="agrarium_close_popup_layer">
    										                        <div class="agrarium_close_button">
    										                            <svg viewBox="0 0 40 40"><path d="M10,10 L30,30 M30,10 L10,30"></path></svg>
    										                        </div>
    										                    </div>
    										                    <div class="agrarium_video_wrapper" data-src="<?php echo esc_url($slide['slide_video_link']); ?>"></div>
    										                </div>
												        </div>
                                             		</div>
                                             <?php }
                                          echo '</div>';
                                      echo '</div>';
                                  echo '</div>';
                              echo '</div>';
                        echo '</div>';
                    }
                    ?>
                </div>                
                <?php 
                	if('yes' === $settings['show_arrows_status']) { ?>
                		<div class="slick-navigation">
	                    <div class="slick-prev slick-arrow"><?php esc_html_e('Previous', 'agrarium_plugin'); ?></div>
	                    <div class="slick-next slick-arrow"><?php esc_html_e('Next', 'agrarium_plugin'); ?></div>
	                	</div>
                	<?php } 
                    $bar_class = '';
                    $video_link = $settings['video_link'];
                    $show_button = ('yes' === $settings['show_video_button'] && !empty($video_link));
                    if($show_button) {
                        $bar_class .= 'with_button_bar ';
                    }
                    if('yes' === $settings['show_dots_status'] && count($slides) > 1) {
                        $bar_class .= 'with_dots_bar';
                    }
                ?>         
                <?php      
                    if('yes' === $settings['show_dots_status'] && count($slides) > 1 ||
                        $show_button == true ||
                        ('yes' === $settings['show_phone_status'] && !empty(agrarium_get_prefered_option('top_bar_contacts_phone'))) ||
                        ('yes' === $settings['show_email_status'] && !empty(agrarium_get_prefered_option('top_bar_contacts_email')))
                    ) { ?>
                		<div class="agrarium_content_slider_bar <?php echo $bar_class;?>">
                			<?php 
                                if('yes' === $settings['show_dots_status'] && count($slides) > 1) {
                                    echo '<div class="slick-dots-navigation"></div>';
                                }
                                if('yes' === $settings['show_phone_status'] && !empty(agrarium_get_prefered_option('top_bar_contacts_phone'))) { ?>
                                    <div class="agrarium_content_slider_phone">
                                        <div class="contact-item">
                                            <?php 
                                                if($contact_phone_title !== '') {
                                                    echo '<p class="contact-item-title">' . esc_html($contact_phone_title) . '</p>';
                                                }
                                            ?>                                            
                                            <span><?php echo esc_html(agrarium_get_prefered_option('top_bar_contacts_phone'))?></span>
                                        </div>
                                    </div>
                                <?php }
                                if('yes' === $settings['show_email_status'] && !empty(agrarium_get_prefered_option('top_bar_contacts_email'))) { ?>
                                    <div class="agrarium_content_slider_email">
                                        <div class="contact-item">
                                            <?php 
                                                if($contact_email_title !== '') {
                                                    echo '<p class="contact-item-title">' . esc_html($contact_email_title) . '</p>';
                                                }
                                            ?>
                                            <span><?php echo esc_html(agrarium_get_prefered_option('top_bar_contacts_email'))?></span>
                                        </div>
                                    </div>
                                <?php }            				
                				if($show_button) { ?>
                  					<div class="agrarium_video_widget slider_video_button">
        					            <div class="agrarium_preview_container">   
        					                <a class="agrarium_video_trigger_button" href="<?php echo esc_js('javascript:void(0)'); ?>">
        					                    <span class="agrarium_button_icon">
                                                    <svg aria-hidden="true" class="progress" width="70" height="70" viewBox="0 0 70 70">
                                                        <path class="progress__circle" d="m35,2.5c17.955803,0 32.5,14.544199 32.5,32.5c0,17.955803 -14.544197,32.5 -32.5,32.5c-17.955803,0 -32.5,-14.544197 -32.5,-32.5c0,-17.955801 14.544197,-32.5 32.5,-32.5z"></path>
                                                        <path class="progress__path" d="m35,2.5c17.955803,0 32.5,14.544199 32.5,32.5c0,17.955803 -14.544197,32.5 -32.5,32.5c-17.955803,0 -32.5,-14.544197 -32.5,-32.5c0,-17.955801 14.544197,-32.5 32.5,-32.5z" pathLength="1"></path>
                                                    </svg>
                                                    <i class="icomoon-play"></i>
                                                </span>
        					                    <?php
        					                    if ($settings['video_button_text'] !== '') {
        					                        ?>
        					                        <span class="agrarium_button_text"><?php echo esc_html($settings['video_button_text']); ?></span>
        					                        <?php
        					                    }
        					                    ?>
        					                </a>
        					            </div>
        				                <div class="agrarium_video_container">
        				                    <div class="agrarium_close_popup_layer">
        				                        <div class="agrarium_close_button">
        				                            <svg viewBox="0 0 40 40"><path d="M10,10 L30,30 M30,10 L10,30"></path></svg>
        				                        </div>
        				                    </div>
        				                    <div class="agrarium_video_wrapper" data-src="<?php echo esc_url($video_link); ?>"></div>
        				                </div>
        						     </div>
                				<?php
                				}                				
                			?>
                      	</div>
                    <?php }
                ?>
                <?php 
                    if('yes' === $settings['show_slider_overlay']) {
                        $img_src = !empty($settings['slider_overlay_image']['id']) ? wp_get_attachment_image_url( $settings['slider_overlay_image']['id'], 'full' ) : get_template_directory_uri() . '/img/page-title-layer.png';
                        $image_meta     = agrarium_get_attachment_meta($settings['slider_overlay_image']['id']);
                        $image_alt_text = $image_meta['alt'];
                        echo '<div class="slider_bottom_overlay"><img src="' . esc_url($img_src) . '" alt="' . esc_attr($image_alt_text) .'"></div>';
                    }
                ?>
                <?php 
                    if($settings['show_scrolldown_status'] === 'yes') { 
                        $scrolldown_link = $settings['scrolldown_link'];
                        ?>
                        <div class="scrolldown_button">
                            <a class="scrolldown_button_anchor" href="<?php echo '#' . esc_attr($scrolldown_link);?>">
                                <svg width="14px" height="30px"><path fill-rule="evenodd"  fill="rgb(62, 59, 47)" d="M13.889,23.22 L7.999,28.911 L7.999,29.999 L5.999,29.999 L5.999,28.911 L0.110,23.22 L1.525,21.608 L5.999,26.83 L5.999,0.0 L7.999,0.0 L7.999,26.83 L12.474,21.608 L13.889,23.22 Z"/>
                                </svg>
                            </a>                            
                        </div>
                    <?php }
                ?>
            </div>
        </div>
        <?php
    }

    protected function content_template() {}

    public function render_plain_content() {}
}
